/* --------------------------------------------------------------
  MyStylesView.js 2019-06-07
  Gambio GmbH
  http://www.gambio.de
  Copyright (c) 2019 Gambio GmbH
  Released under the GNU General Public License (Version 2)
  [http://www.gnu.org/licenses/gpl-2.0.html]
  --------------------------------------------------------------*/


/* globals -Modal */

'use strict';

import Ajax from '../libs/Ajax';
import Modal from '../libs/Modal';
import Router from '../libs/Router';
import EventsEmitter from '../libs/EventsEmitter';
import StyleApi from '../api/StyleApi';
import UserApi from '../api/UserApi';
import BoilerplateApi from '../api/BoilerplateApi';
import CacheApi from '../api/CacheApi';

/**
 * MyStylesView Controller
 */
export default class MyStylesView {
	/**
	 * Class Constructor
	 *
	 * @param {String} selectedStyleName Optional (null), whether to select a specific style from the list.
	 */
	constructor(selectedStyleName = null) {
		/**
		 * Selected Style Name
		 *
		 * @type {String}
		 */
		this.selectedStyleName = selectedStyleName;
	}
	
	/**
	 * Initialize Controller
	 *
	 * @return {jQuery.Promise} Will be resolved once the method is ready.
	 */
	initialize() {
		const deferred = $.Deferred();
		
		StyleApi.list().done(styles => {
			this._loadViewTemplate(styles.config);
			this._bindEventHandlers();
			EventsEmitter.triggerControllerInitialized($('.style-edit-view'), ['MyStylesView']);
			deferred.resolve();
		});
		
		return deferred.promise();
	}
	
	/**
	 * Destroy Controller
	 *
	 * @return {jQuery.Promise} Will be resolved once the method is ready.
	 */
	destroy() {
		const deferred = $.Deferred();
		const $styleEditView = $('.style-edit-view');
		
		$('.styles-container').perfectScrollbar('destroy');
		
		$styleEditView.fadeOut(300, () => {
			$styleEditView.html(''); // Remove the HTML code.
			EventsEmitter.triggerControllerDestroyed($styleEditView, ['MyStylesView']);
			deferred.resolve();
		});
		
		$('.my-styles-list, .styles-list').off();
		
		return deferred.promise();
	}
	
	/**
	 * Event: Click the create new style button.
	 *
	 * @private
	 */
	_onClickCreateStyle() {
		Router.load('CreateStyleModal');
	}
	
	/**
	 * Event: Click the upload style button.
	 *
	 * @private
	 */
	_onClickUploadStyle() {
		Router.load('UploadStyleModal');
	}
	
	/**
	 * Event: Click the terminate button.
	 *
	 * @private
	 */
	_onClickTerminate() {
		UserApi.logout().done(() => {
			this.destroy();
			EventsEmitter.triggerAppTerminated($('.style-edit-view'), ['MyStylesView']);
			window.location.replace(StyleEdit.Config.get('previewUrl'));
		});
	}
	
	/**
	 * Activate Configuration
	 *
	 * @param {jQuery.Event} event
	 * 
	 * @private
	 */
	_onClickActivateStyle(event) {
		const $element = $(event.currentTarget);
		const styleName = $element.parents('.element').find('.info .name').text();
		
		StyleApi.activate(styleName).done((styles) => {
			EventsEmitter.triggerStyleActivated($element, [styleName]);
			this._loadViewTemplate(styles);
			this._bindEventHandlers();
			
			// Now we have a new standard style. So let's renew the CSS cache ...
			let url = $('#main-css', $('.style-edit-preview iframe').contents()).attr('href');
			
			if (url.indexOf('&style_name=') > -1) {
				url = url.replace(/&style_name=.*/g, '');
			}
			
			if (url.indexOf('&renew_cache=1') === -1) {
				url += '&renew_cache=1';
			}
			
			Ajax.get(StyleEdit.Config.get('previewUrl') + '/' + url).done(() => {
				const parameter = '?style_edit_style_name=' + encodeURIComponent(styleName);
				const url = StyleEdit.Config.get('previewUrl') + parameter;
				$('.style-edit-preview iframe').attr('src', url);
				EventsEmitter.triggerPreviewReloaded($('.style-edit-view'), [styleName]);
			});
		});
	}
	
	/**
	 * Event: Edit Configuration
	 *
	 * This method must open the edit settings view.
	 *
	 * @param {jQuery.Event} event
	 * 
	 * @private
	 */
	_onClickEditStyle(event) {
		const styleName = $(event.currentTarget).parents('.element').find('.info .name').text();
		
		this.destroy().done(() => {
			Router.load('EditStyleView', [styleName]);
		});
	}
	
	/**
	 * Event: Duplicate Configuration
	 *
	 * @param {jQuery.Event} event
	 *
	 * @private
	 */
	_onClickDuplicateStyle(event) {
		Router.load('DuplicateStyleModal', [$(event.currentTarget).parents('.element').find('.info .name').text()]);
	}
	
	/**
	 * Delete Configuration
	 *
	 * @param {jQuery.Event} event
	 *
	 * @private
	 */
	_onClickDeleteStyle(event) {
		const $element = $(event.currentTarget);
		const title = StyleEdit.Language.translate('title_delete_style_modal', 'style_edit');
		const message = StyleEdit.Language.translate('message_delete_style_modal', 'style_edit');
		
		Modal.prompt(title, message).done(() => {
			const styleName = $element.parents('.element').find('.info .name').text();
			
			StyleApi.delete(styleName).done((styles) => {
				EventsEmitter.triggerStyleDeleted($element, [styleName]);
				this._loadViewTemplate(styles);
				this._bindEventHandlers();
			});
		});
	}
	
	/**
	 * Event: Style Selected
	 *
	 * This callback must refresh the template preview based on the selected style.
	 *
	 * @param {jQuery.Event} event
	 *
	 * @private
	 */
	_onClickStyleElement(event) {
		const $element = $(event.currentTarget);
		
		if (event.target === $element.find('.dropdown-icon i').get(0) 
			|| event.target === $element.find('.dropdown-icon i').get(1)) {
			return; // The user just clicked in the dropdown icon, so do not refresh the iframe.
		}
		
		$('.styles-list .element.selected').removeClass('selected');
		$element.addClass('selected');
		
		const styleName = $(this).find('.name').text();
		const parameter = '?style_edit_style_name=' + encodeURIComponent($element.find('.name').text());
		const url = StyleEdit.Config.get('previewUrl') + parameter;
		const iframeSrc = $('.style-edit-preview iframe').attr('src');
		const isActive = $element.find('.status:visible').length > 0;
		
		if (iframeSrc === url || (isActive && iframeSrc === StyleEdit.Config.get('previewUrl'))) {
			return; // The old preview URL and the new are the same so do not perform a reload.
		}
		
		$('.style-edit-preview iframe').attr('src', url);
		EventsEmitter.triggerPreviewReloaded($('.style-edit-view'), [styleName]);
	}
	
	/**
	 * Event: New Style Created
	 *
	 * This handler will trigger the list refresh. The new style must also be selected
	 * after the list has completed loading.
	 *
	 * @param {jQuery.Event} event
	 *
	 * @private
	 */
	_onStyleCreated(event, response) {
		const mainCssUrl = StyleEdit.Config.get('previewUrl') + '/'
				+ $('#main-css', $('.style-edit-preview iframe').contents()).attr('href');
		
		CacheApi.renew(mainCssUrl, response.name).done(() => {
			StyleApi.list().done(styles => {
				this._loadViewTemplate(styles.config);
				this._bindEventHandlers();
				$('.my-styles-view .element .name').filter((index, element) => {
					return $(element).text() === response.name;
				}).click();
			});
		});
	}
	
	/**
	 * Event: Download Style
	 *
	 * @param {jQuery.Event} event
	 * 
	 * @private
	 */
	_onClickDownloadStyle(event) {
		const url = StyleEdit.Config.get('baseUrl') + '/api.php/templates/'
			+ StyleEdit.Config.get('template') + '/download/'
			+ encodeURIComponent($(event.currentTarget).parents('.element').find('.name').text())
			+ (StyleEdit.Config.get('theme') ? '?theme' : '');
		window.open(url, '_blank');
	}
	
	/**
	 * Event: Dropdown Toggle
	 *
	 * This event handler is used for updating the scroll bars.
	 *
	 * @private
	 */
	_onDropdownToggle() {
		if ($('.styles-container').perfectScrollbar) {
			$('.styles-container').perfectScrollbar('update');
		}
	}
	
	/**
	 * Event: Window Resize
	 *
	 * @private
	 */
	_onWindowResize() {
		this._resizeStylesContainer();
	}
	
	/**
	 * Resize styles container.
	 *
	 * @private
	 */
	_resizeStylesContainer() {
		const stylesContainerHeight = window.innerHeight
			- $('.style-edit-headline').outerHeight()
			- $('.my-styles-view .footer-container').outerHeight();
		$('.my-styles-view .styles-container').height(stylesContainerHeight);
	}
	
	/**
	 * Load view template.
	 *
	 * @param {Object[]} styles Contains the existing style configurations data for the list.
	 *
	 * @private
	 */
	_loadViewTemplate(styles) {
		const template = $('#my-styles-view-template').html();
		const data = {
			title_my_styles_view: StyleEdit.Language.translate('title_my_styles_view', 'style_edit'),
			message_no_styles: StyleEdit.Language.translate('message_no_styles', 'style_edit'),
			option_new_style: StyleEdit.Language.translate('option_new_style', 'style_edit'),
			option_upload_style: StyleEdit.Language.translate('option_upload_style', 'style_edit'),
			option_terminate: StyleEdit.Language.translate('option_terminate', 'style_edit'),
			availableStyles: styles,
			label_active: StyleEdit.Language.translate('label_active', 'style_edit'),
			option_activate: StyleEdit.Language.translate('option_activate', 'style_edit'),
			option_edit: StyleEdit.Language.translate('option_edit', 'style_edit'),
			option_duplicate: StyleEdit.Language.translate('option_duplicate', 'style_edit'),
			option_delete: StyleEdit.Language.translate('option_delete', 'style_edit'),
			option_download: StyleEdit.Language.translate('option_download', 'style_edit')
		};
		
		// Add language translations to the styles.
		const dateFormat = StyleEdit.Config.get('languageCode') === 'de' ? 'dd.MM.yyyy' : 'MM.dd.yyyy';
		let tempStyleIndex = -1;
		
		$.each(data.availableStyles, (index, style) => {
			if (style.name === StyleEdit.Config.get('tempStyleName')) {
				tempStyleIndex = index;
				return true; // continue
			}
			style.modificationDate = Date.parse(style.modificationDate).toString(dateFormat);
		});
		
		// Remove the temporary style config from the array or create it if it doesn't exist.
		if (tempStyleIndex > -1) {
			data.availableStyles.splice(tempStyleIndex, 1);
		} else {
			this._createTemporaryStyle();
		}
		
		$('.style-edit-view')
			.html(Mustache.render(template, data))
			.fadeIn()
			.find('.styles-list')
			.off('click')
			.on('click', '.action.activate', event => this._onClickActivateStyle(event))
			.on('click', '.action.edit', event => this._onClickEditStyle(event))
			.on('click', '.action.duplicate', event => this._onClickDuplicateStyle(event))
			.on('click', '.action.delete', event => this._onClickDeleteStyle(event))
			.on('click', '.action.download', event => this._onClickDownloadStyle(event))
			.on('shown.bs.dropdown hidden.bs.dropdown', '.element', () => this._onDropdownToggle());
		
		// Explicitly select the active element upon initialization. 
		const styleName = this.selectedStyleName || $('.style-edit-view .element.active .name').text();
		const parameter = '?style_edit_style_name=' + encodeURIComponent(styleName);
		
		$('.style-edit-preview iframe').attr('src', StyleEdit.Config.get('previewUrl') + parameter);
		$('.style-edit-view .element.selected').removeClass('selected');
		$('.style-edit-view .element .name').filter((index, element) => {
			return $(element).text() === styleName;
		}).parents('.element').addClass('selected');
		
		// Add material design effects to new objects. 
		$.material.init();
		
		// Add scrollbars.
		this._resizeStylesContainer();
		$('.styles-container').perfectScrollbar(); 
	}
	
	/**
	 * Create the temporary style config.
	 *
	 * The temporary style configuration is used whenever the user wants to see his current changes
	 * in the edit-style-view as a workaround to the current PHP API. That means that whenever StyleEdit
	 * is initialized it is required that this file is present on the server.
	 *
	 * The temporary configuration file must not be displayed in the styles list, it is used only internally.
	 *
	 * You can get the temporary style name from the config object: StyleEdit.Config.get('tempStyleName').
	 *
	 * @private
	 */
	_createTemporaryStyle() {
		BoilerplateApi.list().done((boilerplates) => {
			if (boilerplates.config.length === 0) {
				throw new Error('Unexpected Error: The current template does not have any boilerplate configuration '
					+ 'files.');
			}
			
			StyleApi.create(boilerplates.config[0].name, StyleEdit.Config.get('tempStyleName'));
		});
	}
	
	/**
	 * Bind element event handlers.
	 *
	 * @private
	 */
	_bindEventHandlers() {
		$('.my-styles-view')
			.off('click')
			.on('click', '.element', event => this._onClickStyleElement(event))
			.on('click', '.btn.create-style', () => this._onClickCreateStyle())
			.on('click', '.btn.upload-style', () => this._onClickUploadStyle())
			.on('click', '.btn.terminate', () => this._onClickTerminate());
		
		$('body')
			.off('StyleEdit.StyleCreated')
			.on('StyleEdit.StyleCreated', (event, response) => this._onStyleCreated(event, response));
		
		$(window)
			.off('resize')
			.on('resize',  () => this._onWindowResize());
	}
}

